/* ------------------------------------------------------------------------------
  File: chr6dm_config.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Function definitions for AHRS configuration
------------------------------------------------------------------------------ */ 
#include <math.h>
#include "stm32f10x.h"
#include "UM6_config.h"
#include "UM6_usart.h"
#include "CHR_matrix.h"
#include "UM6_states.h"

// Global structure storing IMU configuration parameters
UM6_config gConfig;
UM6_data gData;

uint16_t gZeroGyroEnable;
int32_t gZeroGyroAverages[3];
uint16_t gZeroGyroSampleCount;
uint16_t gGyrosCalibrated;

uint16_t gSelfTestEnabled;
uint16_t gSelfTestSamplesIgnored;


/*******************************************************************************
* Function Name  : GetConfiguration
* Input          : None
* Output         : None
* Return         : None
* Description    : Fills the gConfig structure with IMU configuration data.
						  If configuration data has been written to flash, then load it.
						  If not, then use factory defaults.
*******************************************************************************/
void GetConfiguration( )
{
	 // If flash has not been programmed yet, then use default configuration.  Otherwise, load configuration from flash
	 if( FGET_FLASH_UNINITIALIZED() )
	 {
		  ResetToFactory();
	 }
	 else
	 {
		  LoadConfigFromFLASH( UM6_USE_CONFIG_ADDRESS );
	 }
	 
	 CopyConfigArrayToStates();
}

/*******************************************************************************
* Function Name  : ClearGlobalData
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Zeroes out all data in the gData.r[] array.
*******************************************************************************/
void ClearGlobalData()
{
	 int i;
	 
	 for( i = 0; i <  DATA_ARRAY_SIZE; i++ )
	 {
		  gData.r[i] = 0;
	 }
}

/*******************************************************************************
* Function Name  : LoadConfigFromFLASH( )
* Input          : None
* Output         : None
* Return         : None
* Description    : Loads the current sensor configuration from FLASH
*******************************************************************************/
void LoadConfigFromFLASH( int flash_address_flag )
{
	 int i;
	 
	 uint32_t flash_start_address;
	 
	 if( flash_address_flag == UM6_USE_CONFIG_ADDRESS )
	 {
		  flash_start_address = FLASH_START_ADDRESS;
	 }
	 else
	 {
		  flash_start_address = FACTORY_FLASH_ADDRESS;
	 }	 

	 for( i = 0; i < CONFIG_ARRAY_SIZE; i++ )
	 {
		  gConfig.r[i] = (uint32_t)( *(__IO uint32_t*)(flash_start_address + 4*i) );
	 }
}


/*******************************************************************************
* Function Name  : WriteConfigurationToFlash
* Input          : None
* Output         : None
* Return         : None
* Description    : Writes the current IMU configuration to flash.
*******************************************************************************/
int32_t WriteConfigurationToFlash( int write_location_flag )
{
	 FLASH_Status FLASHStatus;
	 uint32_t i;
	 
	 uint32_t flash_start_address;
	 
	 if( write_location_flag == UM6_USE_FACTORY_ADDRESS )
	 {
		  flash_start_address = FACTORY_FLASH_ADDRESS;
	 }
	 else
	 {
		  flash_start_address = FLASH_START_ADDRESS;
	 }
	 	 
	 FLASH_Unlock();
	 
	 // Clear all pending flags
	 FLASH_ClearFlag(FLASH_FLAG_BSY | FLASH_FLAG_EOP | FLASH_FLAG_PGERR | FLASH_FLAG_WRPRTERR);
	 
	 // Erase FLASH page in preparation for write operation
	 for (i = 0; i < CONFIG_ARRAY_SIZE; i++ )
	 {
		  FLASHStatus = FLASH_ErasePage(flash_start_address + 4*i);
		  
		  if( FLASHStatus != FLASH_COMPLETE )
		  {
				FLASH_Lock();
				return FLASHStatus;
		  }
	 }
	 
	 // Write configuration data
	 for (i = 0; i < CONFIG_ARRAY_SIZE; i++ )
	 {		  
		  // Write FLASH data
		  FLASHStatus = FLASH_ProgramWord(flash_start_address + 4*i, gConfig.r[i]);
		  
		  if( FLASHStatus != FLASH_COMPLETE )
		  {
				FLASH_Lock();
				return FLASHStatus;
		  }

		  // Make sure new flash memory contents match
		  if( gConfig.r[i] != (uint32_t)( *(__IO uint32_t*)(flash_start_address + 4*i) ) )
		  {
				FLASH_Lock();
				return FLASH_TIMEOUT;
		  }		  
	 }
	 
	 FLASH_Lock();
	 
	 return FLASH_COMPLETE;
	 
}

/*******************************************************************************
* Function Name  : UpdateBroadcastRate
* Input          : None
* Output         : None
* Return         : None
* Description    : Sets the Timer2 period to adjust IMU broadcast frequency.
						  In Broadcast Mode, data is transmitted on every timer interrupt
*******************************************************************************/
void UpdateBroadcastRate( uint8_t new_rate )
{
	 TIM_TimeBaseInitTypeDef  TIM_TimeBaseStructure;
	 uint16_t new_period;
	 	 
	 // Calculate new period.  The desired broadcast frequency is given by
	 // ft = (280/255)*new_rate + 20
	 // which yields rates ranging from 20 Hz to ~ 300 Hz.
	 // With a prescaler value of 100, a system clock of 72Mhz, and no clock
	 // division, the timer period should be:
	 // new_period = 720000/ft
	 new_period = (uint16_t)(720000.0/(1.09804*(float)new_rate + 20.0));

	 // Update TIM2
	 TIM_TimeBaseStructure.TIM_Period = new_period;
	 TIM_TimeBaseStructure.TIM_Prescaler = 100;
	 TIM_TimeBaseStructure.TIM_ClockDivision = 0;
	 TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;

	 TIM_TimeBaseInit(TIM2, &TIM_TimeBaseStructure);
	 	 
}

/*******************************************************************************
* Function Name  : EnableBroadcastMode
* Input          : None
* Output         : None
* Return         : None
* Description    : Turns on broadcast mode and calls UpdateBroadcastRate
*******************************************************************************/
void EnableBroadcastMode( uint8_t new_rate )
{
	 // Set broadcast rate
	 UpdateBroadcastRate( new_rate );
	 
	 // Enable Timer 2
	 TIM_Cmd(TIM2, ENABLE);
	 
	 // Clear pending interrupt bit
	 TIM_ClearITPendingBit(TIM2, TIM_IT_Update);
	 
	 // TIM IT enable
	 TIM_ITConfig(TIM2, TIM_IT_Update, ENABLE);
}

/*******************************************************************************
* Function Name  : DisableBroadcastMode
* Input          : None
* Output         : None
* Return         : None
* Description    : Disables Broadcast Mode by turning off Timer 2
*******************************************************************************/
void DisableBroadcastMode( void )
{ 
	 // Disable Timer 2
	 TIM_Cmd( TIM2, DISABLE );	 
}

/*******************************************************************************
* Function Name  : StartGyroCalibration
* Input          : None
* Output         : None
* Return         : None
* Description    : Sets the gZeroGyroEnable flag, which triggers the FIR filtering
						  code in process_input_buffers to start summing outputs to 
						  obtain an average.  The average will be used as the new
						  zero point.
*******************************************************************************/
void StartGyroCalibration( )
{
	 gZeroGyroSampleCount = 0;
	 gZeroGyroAverages[0] = 0;
	 gZeroGyroAverages[1] = 0;
	 gZeroGyroAverages[2] = 0;

	 gZeroGyroEnable = 1;
}

/*******************************************************************************
* Function Name  : StopGyroCalibration
* Input          : None
* Output         : None
* Return         : None
* Description    : Finishes gyro calibration cycle. This function is called by 
						  FIR filtering software in process_input_buffers after enough
						  data has been collected to make an average.
*******************************************************************************/
void StopGyroCalibration( )
{
	 gZeroGyroEnable = 0;
	 
	 gZeroGyroAverages[0] = (int16_t)(round((float)gZeroGyroAverages[0]/(float)(gZeroGyroSampleCount)));
	 gZeroGyroAverages[1] = (int16_t)(round((float)gZeroGyroAverages[1]/(float)(gZeroGyroSampleCount)));
	 gZeroGyroAverages[2] = (int16_t)(round((float)gZeroGyroAverages[2]/(float)(gZeroGyroSampleCount)));
	 
	 gConfig.r[UM6_GYRO_BIAS_XY] = (gZeroGyroAverages[0] << 16) | (gZeroGyroAverages[1]);
	 gConfig.r[UM6_GYRO_BIAS_Z] = (gZeroGyroAverages[2] << 16);
	 
	 gStateData.beta_p = gZeroGyroAverages[0];
	 gStateData.beta_q = gZeroGyroAverages[1];
	 gStateData.beta_r = gZeroGyroAverages[2];	 
	 
	 gGyrosCalibrated = 1;

	 // Send packet containing gyro zero data
	 SendGlobalData( UM6_GYRO_BIAS_XY, ADDRESS_TYPE_CONFIG, PACKET_IS_BATCH, BATCH_SIZE_2 );
}


/*******************************************************************************
* Function Name  : StartSelfTest
* Input          : None
* Output         : None
* Return         : None
* Description    : Sets the gSelfTestEnable flag, which triggers the self-test
				       sequence on the IMU.
*******************************************************************************/
void StartSelfTest( void )
{
	 gSelfTestEnabled = 1;
	 gSelfTestSamplesIgnored = 0;

	 // TODO: Activate magnetometer bias test
	 
	 // Assert self-test pin
	 // TODO: Figure out which self-test pin to actually use on the UM6
//	 GPIO_WriteBit( GPIOA, GPIO_Pin_13, Bit_SET );
}

/*******************************************************************************
* Function Name  : StopSelfTest
* Input          : None
* Output         : None
* Return         : None
* Description    : 
*******************************************************************************/
void StopSelfTest( uint16_t result )
{
	 // Clear self-test pin
	 // TODO: Figure out which self-test pin to actually use on the UM6
//	 GPIO_WriteBit( GPIOA, GPIO_Pin_13, Bit_RESET );
	 
	 // TODO: Stop magnetometer bias test
	 
	 gSelfTestEnabled = 0;
	 
	 // TODO: write self-test results to the UM6_STATUS register
	 
	 // TODO: send a packet with the test results
}

/*******************************************************************************
* Function Name  : ResetToFactory
* Input          : None
* Output         : None
* Return         : None
* Description    : Resets all AHRS settings to factory default.
*******************************************************************************/
void ResetToFactory( void )
{
	 fConvert f2int;
	 
	 // If flash has not been programmed yet, then use default configuration.  Otherwise, load configuration from flash
	 if( FGET_FACTORY_UNINITIALIZED() )
	 {
		  // Communication configuration
	 //	 gConfig.r[UM6_COMMUNICATION] = UM6_BROADCAST_ENABLED | UM6_ACCELS_PROC_ENABLED | UM6_GYROS_PROC_ENABLED | UM6_MAG_PROC_ENABLED | UM6_EULER_ENABLED;
		  gConfig.r[UM6_COMMUNICATION] =  UM6_BROADCAST_ENABLED | UM6_GYROS_PROC_ENABLED | UM6_ACCELS_PROC_ENABLED | UM6_MAG_PROC_ENABLED | UM6_EULER_ENABLED;
	 //	 gConfig.r[UM6_COMMUNICATION] = UM6_BROADCAST_ENABLED | UM6_MAG_PROC_ENABLED | UM6_MAG_RAW_ENABLED | UM6_GYROS_PROC_ENABLED | UM6_EULER_ENABLED | UM6_COV_ENABLED;
	 //	 gConfig.r[UM6_COMMUNICATION] = UM6_BROADCAST_ENABLED | UM6_GYROS_PROC_ENABLED | UM6_GYROS_RAW_ENABLED;
	 //	 gConfig.r[UM6_COMMUNICATION] = 0;
		  gConfig.r[UM6_COMMUNICATION] |= 0x0000;		// This is the broadcast rate (or, rather, what is used to set the broadcast rate)
		  gConfig.r[UM6_COMMUNICATION] |= (5 << UM6_BAUD_START_BIT);
		  
		  // MISC configuration
		  gConfig.r[UM6_MISC_CONFIG] = UM6_MAG_UPDATE_ENABLED | UM6_ACCEL_UPDATE_ENABLED | UM6_QUAT_ESTIMATE_ENABLED | UM6_GYRO_STARTUP_CAL;
		  
		  // Magnetometer reference vector
		  f2int.float_val = 0.136;
		  gConfig.r[UM6_MAG_REF_X] = f2int.uint32_val;
		  
		  f2int.float_val = 0.4065;
		  gConfig.r[UM6_MAG_REF_Y] = f2int.uint32_val;
		  
		  f2int.float_val = 0.894;
		  gConfig.r[UM6_MAG_REF_Z] = f2int.uint32_val;
		  
		  // Accelerometer reference vector
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_REF_X] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_REF_Y] = f2int.uint32_val;
		  
		  f2int.float_val = -1.0;
		  gConfig.r[UM6_ACCEL_REF_Z] = f2int.uint32_val;
		  
		  // EKF variances
		  f2int.float_val = 2.0;
		  gConfig.r[UM6_EKF_MAG_VARIANCE] = f2int.uint32_val;
		  
		  f2int.float_val = 2.0;
		  gConfig.r[UM6_EKF_ACCEL_VARIANCE] = f2int.uint32_val;
		  
		  f2int.float_val = 0.1;
		  gConfig.r[UM6_EKF_PROCESS_VARIANCE] = f2int.uint32_val;
		  
		  // Gyro biases
		  gConfig.r[UM6_GYRO_BIAS_XY] = 0;
		  gConfig.r[UM6_GYRO_BIAS_Z] = 0;
		  
		  // Accelerometer biases
		  gConfig.r[UM6_ACCEL_BIAS_XY] = 0;
		  gConfig.r[UM6_ACCEL_BIAS_Z] = 0;
		  
		  // Magnetometer biases
		  gConfig.r[UM6_MAG_BIAS_XY] = 0;
		  gConfig.r[UM6_MAG_BIAS_Z] = 0;
		  
		  // Accelerometer alignment matrix
		  f2int.float_val = .0000625;
		  gConfig.r[UM6_ACCEL_CAL_00] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_CAL_01] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_CAL_02] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_CAL_10] = f2int.uint32_val;
		  
		  f2int.float_val = .0000625;
		  gConfig.r[UM6_ACCEL_CAL_11] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_CAL_12] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_CAL_20] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_ACCEL_CAL_21] = f2int.uint32_val;
		  
		  f2int.float_val = .0000625;
		  gConfig.r[UM6_ACCEL_CAL_22] = f2int.uint32_val;
		  
		  // Rate gyro alignment matrix
		  f2int.float_val = .06956;
		  gConfig.r[UM6_GYRO_CAL_00] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_GYRO_CAL_01] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_GYRO_CAL_02] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_GYRO_CAL_10] = f2int.uint32_val;
		  
		  f2int.float_val = .06956;
		  gConfig.r[UM6_GYRO_CAL_11] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_GYRO_CAL_12] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_GYRO_CAL_20] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_GYRO_CAL_21] = f2int.uint32_val;
		  
		  f2int.float_val = .06956;
		  gConfig.r[UM6_GYRO_CAL_22] = f2int.uint32_val;
		  
		  // Magnetometer calibration matrix
		  f2int.float_val = 0.00271;
		  gConfig.r[UM6_MAG_CAL_00] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_MAG_CAL_01] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_MAG_CAL_02] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_MAG_CAL_10] = f2int.uint32_val;
		  
		  f2int.float_val = 0.00271;
		  gConfig.r[UM6_MAG_CAL_11] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_MAG_CAL_12] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_MAG_CAL_20] = f2int.uint32_val;
		  
		  f2int.float_val = 0.0;
		  gConfig.r[UM6_MAG_CAL_21] = f2int.uint32_val;
		  
		  f2int.float_val = 0.00271;
		  gConfig.r[UM6_MAG_CAL_22] = f2int.uint32_val;
	 }
	 else
	 {
		  LoadConfigFromFLASH( UM6_USE_FACTORY_ADDRESS );
	 }
	 
	 CopyConfigArrayToStates();	 
}

/*******************************************************************************
* Function Name  : CopyStatesToDataArray
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Copies the data stored in the gStateData structure (defined in UM6_states.c)
and the gSensorData structure (also defined in UM6_states.c)into the gData 
structure (defined in UM6_config.c).  This is done so that
the communication code can easily retrieve and transmit all data from the
sensor.

*******************************************************************************/
void CopyStatesToDataArray( void )
{
	 fConvert f2int;
	 int16_t converted_data1;
	 int16_t converted_data2;
	 
	 // Raw gyro, accel, and mag data
	 gData.r[UM6_GYRO_RAW_XY-DATA_REG_START_ADDRESS] = (uint32_t)((gSensorData.gyro_x << 16) | (gSensorData.gyro_y & 0x0FFFF));
	 gData.r[UM6_GYRO_RAW_Z-DATA_REG_START_ADDRESS] = (uint32_t)(gSensorData.gyro_z << 16);
	 
	 gData.r[UM6_ACCEL_RAW_XY-DATA_REG_START_ADDRESS] = (uint32_t)((gSensorData.accel_x << 16) | (gSensorData.accel_y & 0x0FFFF));
	 gData.r[UM6_ACCEL_RAW_Z-DATA_REG_START_ADDRESS] = (uint32_t)(gSensorData.accel_z << 16);
	 
	 gData.r[UM6_MAG_RAW_XY-DATA_REG_START_ADDRESS] = (uint32_t)((gSensorData.mag_x << 16) | (gSensorData.mag_y & 0x0FFFF));
	 gData.r[UM6_MAG_RAW_Z-DATA_REG_START_ADDRESS] = (uint32_t)(gSensorData.mag_z << 16);
	 
	 // Processed gyros
	 // The maximum measureable rotation rate is +/-2000 deg/sec.  We have 16 bits to represent it as a signed integer
	 // 2^16/4000 = 16.384
	 converted_data1 = (int16_t)roundf(gStateData.gyro_x*16.384);
	 converted_data2 = (int16_t)roundf(gStateData.gyro_y*16.384);
	 gData.r[UM6_GYRO_PROC_XY-DATA_REG_START_ADDRESS] = (uint32_t)((converted_data1 << 16) | (converted_data2 & 0x0FFFF));
	 
	 converted_data1 = (int16_t)roundf(gStateData.gyro_z*16.384);
	 gData.r[UM6_GYRO_PROC_Z-DATA_REG_START_ADDRESS] = (uint32_t)(converted_data1 << 16);
	 
	 // Processed accels
	 // The maximum measureable acceleration is assumed to be +/- 6g.  We have 16 bits to represent it as a signed integer
	 // 2^16/12 = 5461.33333333
	 converted_data1 = (int16_t)roundf(gStateData.accel_x*5461.33333333);
	 converted_data2 = (int16_t)roundf(gStateData.accel_y*5461.33333333);
	 gData.r[UM6_ACCEL_PROC_XY-DATA_REG_START_ADDRESS] = (uint32_t)((converted_data1 << 16) | (converted_data2 & 0x0FFFF));
	 
	 converted_data1 = (int16_t)roundf(gStateData.accel_z*5461.33333333);
	 gData.r[UM6_ACCEL_PROC_Z-DATA_REG_START_ADDRESS] = (uint32_t)(converted_data1 << 16);	
	 
	 // Processed magnetic sensor data
	 // If the magnetometer is even remotely calibrated correctly, the maximum value of any processed magnetometer axis is
	 // equal to 1.0.  Can't guarantee, though, that calibration will always be good.  instead, assume maximum values of 10.
	 // 2^16/20 = 3276.8
	 converted_data1 = (int16_t)roundf(gStateData.mag_x*3276.8);
	 converted_data2 = (int16_t)roundf(gStateData.mag_y*3276.8);
	 gData.r[UM6_MAG_PROC_XY-DATA_REG_START_ADDRESS] = (uint32_t)((converted_data1 << 16) | (converted_data2 & 0x0FFFF));
	 
	 converted_data1 = (int16_t)roundf(gStateData.mag_z*3276.8);
	 gData.r[UM6_MAG_PROC_Z-DATA_REG_START_ADDRESS] = (uint32_t)(converted_data1 << 16);
	 
	 // Euler angles
	 // Maximum euler angle values are +/360 degrees.
	 // 2^16/(360*2) = 91.0222222222
	 converted_data1 = (int16_t)roundf(gStateData.phi*91.0222222);
	 converted_data2 = (int16_t)roundf(gStateData.theta*91.0222222);
	 gData.r[UM6_EULER_PHI_THETA-DATA_REG_START_ADDRESS] = (uint32_t)((converted_data1 << 16) | (converted_data2 & 0x0FFFF));
	 
	 converted_data1 = (int16_t)roundf(gStateData.psi*91.0222222);
	 gData.r[UM6_EULER_PSI-DATA_REG_START_ADDRESS] = (uint32_t)(converted_data1 << 16);
	 
	 // Quaternions
	 // Quaternions are normalized, so the maximum value of any single element is 1.0.  However, let the value go to +/- 1.1 to 
	 // to avoid overflow problems at 1.0
	 // 2^16/2.2 = 29789.09091
	 converted_data1 = (int16_t)roundf(gStateData.qib.a*29789.09091);
	 converted_data2 = (int16_t)roundf(gStateData.qib.b*29789.09091);
	 gData.r[UM6_QUAT_AB-DATA_REG_START_ADDRESS] = (uint32_t)((converted_data1 << 16) | (converted_data2 & 0x0FFFF));
	 
	 converted_data1 = (int16_t)roundf(gStateData.qib.c*29789.09091);
	 converted_data2 = (int16_t)roundf(gStateData.qib.d*29789.09091);
	 gData.r[UM6_QUAT_CD-DATA_REG_START_ADDRESS] = (uint32_t)((converted_data1 << 16) | (converted_data2 & 0x0FFFF));
	 
	 // Error covariance matrix
	 f2int.float_val = gStateData.Sigma.data[0][0];
	 gData.r[UM6_ERROR_COV_00-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[0][1];
	 gData.r[UM6_ERROR_COV_01-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[0][2];
	 gData.r[UM6_ERROR_COV_02-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[0][3];
	 gData.r[UM6_ERROR_COV_03-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[1][0];
	 gData.r[UM6_ERROR_COV_10-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[1][1];
	 gData.r[UM6_ERROR_COV_11-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[1][2];
	 gData.r[UM6_ERROR_COV_12-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[1][3];
	 gData.r[UM6_ERROR_COV_13-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[2][0];
	 gData.r[UM6_ERROR_COV_20-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[2][1];
	 gData.r[UM6_ERROR_COV_21-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[2][2];
	 gData.r[UM6_ERROR_COV_22-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[2][3];
	 gData.r[UM6_ERROR_COV_23-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[3][0];
	 gData.r[UM6_ERROR_COV_30-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[3][1];
	 gData.r[UM6_ERROR_COV_31-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[3][2];
	 gData.r[UM6_ERROR_COV_32-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 f2int.float_val = gStateData.Sigma.data[3][3];
	 gData.r[UM6_ERROR_COV_33-DATA_REG_START_ADDRESS] = f2int.uint32_val;
	 
	 // Status register
	 // This register needn't be modified because it is always accessed directly.
	 
}

/*******************************************************************************
* Function Name  : UpdateSerialBaud
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Updates the serial baudrate to the baudrate selected in the global configuration
array.

*******************************************************************************/
void UpdateSerialBaud()
{
	 int baud_rate = 115200;
	 
	 uint8_t baud_selection = (gConfig.r[UM6_COMMUNICATION] >> UM6_BAUD_START_BIT) & UM6_BAUD_RATE_MASK;
	 
	 switch(baud_selection)
	 {
		  case 0:
				baud_rate = 9600;
		  break;
		  
		  case 1:
				baud_rate = 14400;
		  break;
		  
		  case 2:
				baud_rate = 19200;
		  break;
		  
		  case 3:
				baud_rate = 38400;
		  break;
		  
		  case 4:
				baud_rate = 57600;
		  break;
		  
		  case 5:
				baud_rate = 115200;
		  break;
		  
		  default:
				baud_rate = 115200;
		  break;
	 }
	 
	 USART1_Configuration( baud_rate );
	 
}

/*******************************************************************************
* Function Name  : CopyConfigArrayToStates
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Copies certain portions of the global configuration array (gConfig, defined in 
UM6_config.c) to the gStateData structure (defined in UM6_states.c).  This is
done for convenience, since many of the configuration options such as gyro and
accel scale factors are actually floating point values.  The gConfig array stores
them as 32-bit unsigned integers, which makes it difficult to access and use them.

This function makes it easier to access floating-point configuration values that
need to be used often by the software.

*******************************************************************************/
void CopyConfigArrayToStates( void )
{
	 fConvert i2f;
	 
	 // Magnetic field reference vector
	 i2f.uint32_val = gConfig.r[UM6_MAG_REF_X-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_ref_x = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_REF_Y-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_ref_y = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_REF_Z-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_ref_z = i2f.float_val;
	 
	 // Accelerometer reference vector
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_REF_X-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_ref_x = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_REF_Y-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_ref_y = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_REF_Z-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_ref_z = i2f.float_val;
	 
	 // Magnetometer and accelerometer variances
	 i2f.uint32_val = gConfig.r[UM6_EKF_MAG_VARIANCE-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_var = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_EKF_ACCEL_VARIANCE-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_var = i2f.float_val;
	 
	 // Process variance matrix
	 i2f.uint32_val = gConfig.r[UM6_EKF_PROCESS_VARIANCE-CONFIG_REG_START_ADDRESS];
	 if( (gConfig.r[UM6_MISC_CONFIG] & UM6_QUAT_ESTIMATE_ENABLED) == 0 )
	 {
		  mat_zero( &gStateData.R, 3, 3 );
		  gStateData.R.data[0][0] = i2f.float_val;
		  gStateData.R.data[1][1] = i2f.float_val;
		  gStateData.R.data[2][2] = i2f.float_val;
		  gStateData.R.data[3][3] = i2f.float_val;
		  
		  gEKF_mode = EKF_MODE_EULER;
	 }
	 else
	 {
		  mat_zero( &gStateData.R, 4, 4 );
		  
		  // Process variance is scaled here so that the performance in Euler Angle mode and Quaternion mode is comparable
		  gStateData.R.data[0][0] = i2f.float_val*0.00001;
		  gStateData.R.data[1][1] = i2f.float_val*0.00001;
		  gStateData.R.data[2][2] = i2f.float_val*0.00001;
		  gStateData.R.data[3][3] = i2f.float_val*0.00001;
		  
		  gEKF_mode = EKF_MODE_QUAT;
	 }
	  
	 gStateData.process_var = i2f.float_val;
	 
	 // Gyro biases
	 gStateData.beta_p = (int16_t)(gConfig.r[UM6_GYRO_BIAS_XY-CONFIG_REG_START_ADDRESS] >> 16);
	 gStateData.beta_q = (int16_t)(gConfig.r[UM6_GYRO_BIAS_XY-CONFIG_REG_START_ADDRESS] & 0x0FFFF);
	 gStateData.beta_r = (int16_t)(gConfig.r[UM6_GYRO_BIAS_Z-CONFIG_REG_START_ADDRESS] >> 16);
	 
	 // Accel biases
	 gStateData.beta_acc_x = (int16_t)(gConfig.r[UM6_ACCEL_BIAS_XY-CONFIG_REG_START_ADDRESS] >> 16);
	 gStateData.beta_acc_y = (int16_t)(gConfig.r[UM6_ACCEL_BIAS_XY-CONFIG_REG_START_ADDRESS] & 0x0FFFF);
	 gStateData.beta_acc_z = (int16_t)(gConfig.r[UM6_ACCEL_BIAS_Z-CONFIG_REG_START_ADDRESS] >> 16);
	 
	 // Mag biases
	 gStateData.beta_mag_x = (int16_t)(gConfig.r[UM6_MAG_BIAS_XY-CONFIG_REG_START_ADDRESS] >> 16);
	 gStateData.beta_mag_y = (int16_t)(gConfig.r[UM6_MAG_BIAS_XY-CONFIG_REG_START_ADDRESS] & 0x0FFFF);
	 gStateData.beta_mag_z = (int16_t)(gConfig.r[UM6_MAG_BIAS_Z-CONFIG_REG_START_ADDRESS] >> 16);
	 
	 // Accelerometer calibration matrix
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_00-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[0][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_01-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[0][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_02-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[0][2] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_10-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[1][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_11-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[1][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_12-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[1][2] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_20-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[2][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_21-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[2][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_ACCEL_CAL_22-CONFIG_REG_START_ADDRESS];
	 gStateData.accel_cal.data[2][2] = i2f.float_val;	

	 gStateData.accel_cal.rows = 3;
	 gStateData.accel_cal.columns = 3;	 
	 
	 // Rate gyro alignment matrix
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_00-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[0][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_01-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[0][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_02-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[0][2] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_10-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[1][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_11-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[1][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_12-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[1][2] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_20-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[2][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_21-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[2][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_GYRO_CAL_22-CONFIG_REG_START_ADDRESS];
	 gStateData.gyro_cal.data[2][2] = i2f.float_val;
	 
	 gStateData.gyro_cal.rows = 3;
	 gStateData.gyro_cal.columns = 3;
	 
	 // Magnetometer calibration matrix
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_00-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[0][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_01-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[0][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_02-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[0][2] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_10-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[1][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_11-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[1][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_12-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[1][2] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_20-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[2][0] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_21-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[2][1] = i2f.float_val;
	 
	 i2f.uint32_val = gConfig.r[UM6_MAG_CAL_22-CONFIG_REG_START_ADDRESS];
	 gStateData.mag_cal.data[2][2] = i2f.float_val;
	 
	 gStateData.mag_cal.rows = 3;
	 gStateData.mag_cal.columns = 3;
}
